import os
from pathlib import Path

from conan import ConanFile
from conan.tools.cmake import CMake, CMakeDeps, CMakeToolchain, cmake_layout
from conan.tools.files import copy, collect_libs, update_conandata, unzip
from conan.tools.scm import Git

from utils import get_lfs_file

class PocoCapsulePackage(ConanFile):
    name = 'poco-capsule'
    license = 'LGPL v3.0'
    description = 'Dependency Injection Framework'
    url = 'https://hxgntech.com/stash/scm/iconfield/pococapsule.git'

    settings = 'os', 'arch', 'compiler', 'build_type'

    options = {
        'shared': [True, False],
        'fPIC': [True, False],
    }

    default_options = {
        'shared': True,
        'fPIC': True,
        'xerces-c/*:shared': False,
    }

    exports = ["__init__.py", "utils/*.py"]


    def set_version(self):
        if not self.version:
            git = Git(self, self.recipe_folder)
            # NOTE: running "git status" updates the index as a side effect and fixes "git describe --dirty" output
            git.run('status')
            self.version = git.run('describe --tags --dirty --broken')


    def export(self):
        git = Git(self, self.recipe_folder)
        scm_url, scm_commit = git.get_url_and_commit()
        update_conandata(self, {"sources": {"commit": scm_commit, "url": scm_url}})


    def source(self):
        git = Git(self)
        sources = self.conan_data["sources"]
        git.clone(url=sources["url"], target=".")
        git.checkout(commit=sources["commit"])


    def layout(self):
        cmake_layout(self)


    def config_options(self):
        if self.settings.os != 'Linux':
            del self.options.fPIC


    def configure(self):
        if 'Windows' not in self.settings.os:
            self.options['*'].fPIC = self.options.get_safe('fPIC') or self.options.shared


    def generate(self):
        deps = CMakeDeps(self)
        deps.generate()

        tc = CMakeToolchain(self)
        tc.generate()

        if 'Windows' in self.settings.os:
            jre_name = "OpenJDK8U-jre_x64_windows_hotspot_8u392b08.zip"
            get_lfs_file.download("hxgntech.com:7099/vs/snapshots.git", "PocoCapsule", jre_name, self.build_folder)
            unzip(self, os.path.join(self.build_folder, jre_name), self.build_folder, strip_root=False)
            jdk_name = "OpenJDK8U-jdk_x64_windows_hotspot_8u392b08.zip"
            get_lfs_file.download("hxgntech.com:7099/vs/snapshots.git", "PocoCapsule", jdk_name, self.build_folder)
            unzip(self, os.path.join(self.build_folder, jdk_name), self.build_folder, strip_root=False)


    def requirements(self):
        self.requires("xerces-c/3.2.3@cci/stable")

    def build(self):
        definitions = dict()
        definitions["CMAKE_BUILD_TYPE"] = self.settings.build_type
        if 'Windows' in self.settings.os:
            definitions["JAVA_HOME"] = os.path.join(self.build_folder, "jdk8u392-b08").replace("\\", "/")

        cmake = CMake(self)
        cmake.configure(definitions)
        cmake.build()


    def package(self):
        copy(self, "LICENSE", self.source_folder, os.path.join(self.package_folder, "licenses"))
        cmake = CMake(self)
        cmake.install()
        if 'Windows' in self.settings.os:
            copy(self, "*", os.path.join(self.build_folder, "jdk8u392-b08-jre"), os.path.join(self.package_folder, "jre"), keep_path=True)


    def package_info(self):
        self.cpp_info.libs = collect_libs(self)
        self.cpp_info.set_property("cmake_build_modules", ["cmake/PocoCapsule.cmake",])
        self.cpp_info.builddirs.append("cmake")
        if 'Windows' in self.settings.os:
            self.buildenv_info.append("JAVA_HOME", os.path.join(self.package_folder, "jre").replace("\\", "/"))
            self.runenv_info.append("JAVA_HOME", os.path.join(self.package_folder, "jre").replace("\\", "/"))
