// +-------------------------------------------------------------------------
// | main.cpp
// | 
// | Author: Gilbert Bernstein
// +-------------------------------------------------------------------------
// | COPYRIGHT:
// |    Copyright Gilbert Bernstein 2013
// |    See the included COPYRIGHT file for further details.
// |    
// |    This file is part of the Cork library.
// |
// |    Cork is free software: you can redistribute it and/or modify
// |    it under the terms of the GNU Lesser General Public License as
// |    published by the Free Software Foundation, either version 3 of
// |    the License, or (at your option) any later version.
// |
// |    Cork is distributed in the hope that it will be useful,
// |    but WITHOUT ANY WARRANTY; without even the implied warranty of
// |    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// |    GNU Lesser General Public License for more details.
// |
// |    You should have received a copy 
// |    of the GNU Lesser General Public License
// |    along with Cork.  If not, see <http://www.gnu.org/licenses/>.
// +-------------------------------------------------------------------------

// This file contains a command line program that can be used
// to exercise Cork's functionality without having to write
// any code.

#include "cork.h"
#include "file_formats/files.h"
#include "util/prelude.h"

#include <iostream>
#include <sstream>
#include <stdexcept>
#include <functional>


using std::cout;
using std::cerr;
using std::endl;
using std::stringstream;
using std::string;
using std::ostream;
using namespace Cork;

class CmdList {
public:
    CmdList();
    ~CmdList() {}
    
    void regCmd(
        string name,
        string helptxt,
        std::function< void(std::vector<string>::iterator &,
                            const std::vector<string>::iterator &) > body
    );
    
    void printHelp(ostream &out);
    void runCommands(std::vector<string>::iterator &arg_it,
                     const std::vector<string>::iterator &end_it);
    
private:
    struct Command {
        string name;    // e.g. "show" will be invoked with option "-show"
        string helptxt; // lines to be displayed
        std::function< void(std::vector<string>::iterator &,
                       const std::vector<string>::iterator &) > body;
    };
    std::vector<Command> commands;
};

CmdList::CmdList()
{
    regCmd("help",
    "-help                  show this help message",
    [this](std::vector<string>::iterator &,
           const std::vector<string>::iterator &) {
        printHelp(cout);
        exit(0);
    });
}

void CmdList::regCmd(
    string name,
    string helptxt,
    std::function< void(std::vector<string>::iterator &,
                        const std::vector<string>::iterator &) > body
) {
    Command cmd = {
        name,
        helptxt,
        body
    };
    commands.push_back(cmd);
}

void CmdList::printHelp(ostream &out)
{
    out <<
    "Welcome to Cork.  Usage:" << endl <<
    "  > cork [-command arg0 arg1 ... argn]*" << endl <<
    "for example," << endl <<
    "  > cork -union box0.off box1.off result.off" << endl <<
    "Options:" << endl;
    for(auto &cmd : commands)
        out << cmd.helptxt << endl;
    out << endl;
}

void CmdList::runCommands(std::vector<string>::iterator &arg_it,
                          const std::vector<string>::iterator &end_it)
{
    while(arg_it != end_it) {
        string arg_cmd = *arg_it;
        if(arg_cmd[0] != '-') {
            cerr << arg_cmd << endl;
            cerr << "All commands must begin with '-'" << endl;
            exit(1);
        }
        arg_cmd = arg_cmd.substr(1);
        arg_it++;
        
        bool found = true;
        for(auto &cmd : commands) {
            if(arg_cmd == cmd.name) {
                cmd.body(arg_it, end_it);
                found = true;
                break;
            }
        }
        if(!found) {
            cerr << "Command -" + arg_cmd + " is not recognized" << endl;
            exit(1);
        }
    }
}

std::function<
    void(std::vector<string>::iterator&, const std::vector<string>::iterator&)>
genericBinaryOp(
    std::function<CorkRawMesh(CorkRawMesh&, CorkRawMesh&, const int)> binop,
    const int nTries)
{
    return [binop, nTries](
               std::vector<string>::iterator& args,
               const std::vector<string>::iterator& end) {
        // data...
        CorkRawMesh in0;
        CorkRawMesh in1;

        if(args == end)
        {
            cerr << "too few args" << endl;
            exit(1);
        }
        in0 = loadMesh(*args);
        args++;

        if(args == end)
        {
            cerr << "too few args" << endl;
            exit(1);
        }
        in1 = loadMesh(*args);
        args++;

        in0 = binop(in0, in1, nTries);

        if(args == end)
        {
            cerr << "too few args" << endl;
            exit(1);
        }
        saveMesh(*args, in0);
        args++;
    };
}

int main(int argc, char *argv[])
{
    initRand(); // that's useful
    
    if(argc < 2) {
        cout << "Please type 'cork -help' for instructions" << endl;
        exit(0);
    }
    
    // store arguments in a standard container
    std::vector<string> args(argc);
    for(int k=0; k<argc; k++) {
        args[k] = argv[k];
    }
    
    auto arg_it = args.begin();
    // *arg_it is the program name to begin with, so advance!
    arg_it++;
    
    CmdList cmds;
    
    // add cmds
    cmds.regCmd("solid",
    "-solid in              Determine whether the input mesh represents\n"
    "                       a solid object.  (aka. watertight) (technically\n"
    "                         solid == closed and non-self-intersecting)",
    [](std::vector<string>::iterator &args,
       const std::vector<string>::iterator &end) {
        CorkRawMesh in;
        if(args == end) { cerr << "too few args" << endl; exit(1); }
        string filename = *args;
        in = loadMesh(*args);
        args++;
        bool solid = isSolid(in);
        cout << "The mesh " << filename << " is: " << endl;
        cout << "    " << ((solid)? "SOLID" : "NOT SOLID") << endl;
    });
    const int nTries = 15;
    cmds.regCmd(
        "union",
        "-union in0 in1 out     Compute the Boolean union of in0 and in1,\n"
        "                       and output the result",
        genericBinaryOp(computeUnion, nTries));
    cmds.regCmd(
        "diff",
        "-diff in0 in1 out      Compute the Boolean difference of in0 and in1,\n"
        "                       and output the result",
        genericBinaryOp(computeDifference, nTries));
    cmds.regCmd(
        "isct",
        "-isct in0 in1 out      Compute the Boolean intersection of in0 and in1,\n"
        "                       and output the result",
        genericBinaryOp(computeIntersection, nTries));
    cmds.regCmd(
        "xor",
        "-xor in0 in1 out       Compute the Boolean XOR of in0 and in1,\n"
        "                       and output the result\n"
        "                       (aka. the symmetric difference)",
        genericBinaryOp(computeSymmetricDifference, nTries));
    cmds.regCmd(
        "resolve",
        "-resolve in0 in1 out   Intersect the two meshes in0 and in1,\n"
        "                       and output the connected mesh with those\n"
        "                       intersections made explicit and connected",
        genericBinaryOp(resolveIntersections, nTries));

    cmds.runCommands(arg_it, args.end());
    
    return 0;
}









