// +-------------------------------------------------------------------------
// | cork.h
// |
// | Author: Gilbert Bernstein
// +-------------------------------------------------------------------------
// | COPYRIGHT:
// |    Copyright Gilbert Bernstein 2013
// |    See the included COPYRIGHT file for further details.
// |
// |    This file is part of the Cork library.
// |
// |    Cork is free software: you can redistribute it and/or modify
// |    it under the terms of the GNU Lesser General Public License as
// |    published by the Free Software Foundation, either version 3 of
// |    the License, or (at your option) any later version.
// |
// |    Cork is distributed in the hope that it will be useful,
// |    but WITHOUT ANY WARRANTY; without even the implied warranty of
// |    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// |    GNU Lesser General Public License for more details.
// |
// |    You should have received a copy
// |    of the GNU Lesser General Public License
// |    along with Cork.  If not, see <http://www.gnu.org/licenses/>.
// +-------------------------------------------------------------------------
#pragma once

#include "rawmesh/rawMesh.h"

#include "cork_export.h"

#include <string>

namespace Cork
{

struct CORK_EXPORT BoolVertexData
{};

struct CORK_EXPORT BoolTriangleData
{
    // internal use by algorithm: copy value when the triangle is subdivided
    std::uint8_t bool_alg_data;
};

// in order to perform remeshing, VertData and TriData must support
struct CORK_EXPORT RemeshVertexData
{
    bool manifold; // whether this point is manifold.
                   // useful for modifying interpolation behavior
    // specify how to compute new data for a vertex in the event of
    // either an edge collapse (via merge) or edge split (via interpolate)
};

template <class VertData, class TriData>
struct CORK_EXPORT SubdivideTriInput
{
    TriData* pt;
    VertData* pv[3];
    VertData* v[3];
};

struct CORK_EXPORT CorkVertex
    : public MinimalVertexData
    , public RemeshVertexData
    , public BoolVertexData
{
    CorkVertex();
    CorkVertex(const MinimalVertexData& from);
    void isctInterpolate(const CorkVertex& v0, const CorkVertex& v1);
};

struct CORK_EXPORT CorkTriangle
    : public MinimalTriangleData
    , public BoolTriangleData
{
    CorkTriangle();
    CorkTriangle(const MinimalTriangleData& from);
    void subdivide(SubdivideTriInput<CorkVertex, CorkTriangle> input);
};

typedef RawMesh<CorkVertex, CorkTriangle> CorkRawMesh;

CORK_EXPORT CorkRawMesh loadMesh(const std::string& filename);

CORK_EXPORT void saveMesh(const std::string& filename, const CorkRawMesh& mesh);

// This function will test whether or not a mesh is solid
CORK_EXPORT bool isSolid(const CorkRawMesh& rawMesh);

// Boolean operations follow
// result = A U B
CORK_EXPORT CorkRawMesh computeUnion(
    const CorkRawMesh& inOut0,
    const CorkRawMesh& in1,
    const int nTries);

// result = A - B
CORK_EXPORT CorkRawMesh computeDifference(
    const CorkRawMesh& in0,
    const CorkRawMesh& in1,
    const int nTries);

// result = A ^ B
CORK_EXPORT CorkRawMesh computeIntersection(
    const CorkRawMesh& in0,
    const CorkRawMesh& in1,
    const int nTries);

// result = A XOR B
CORK_EXPORT CorkRawMesh computeSymmetricDifference(
    const CorkRawMesh& in0,
    const CorkRawMesh& in1,
    const int nTries);

// Not a Boolean operation, but related:
//  No portion of either surface is deleted.  However, the
//  curve of intersection between the two surfaces is made explicit,
//  such that the two surfaces are now connected.
CORK_EXPORT CorkRawMesh resolveIntersections(
    const CorkRawMesh& in0,
    const CorkRawMesh& in1,
    const int nTries);

} // namespace Cork
