// +-------------------------------------------------------------------------
// | cork.cpp
// |
// | Author: Gilbert Bernstein
// +-------------------------------------------------------------------------
// | COPYRIGHT:
// |    Copyright Gilbert Bernstein 2013
// |    See the included COPYRIGHT file for further details.
// |
// |    This file is part of the Cork library.
// |
// |    Cork is free software: you can redistribute it and/or modify
// |    it under the terms of the GNU Lesser General Public License as
// |    published by the Free Software Foundation, either version 3 of
// |    the License, or (at your option) any later version.
// |
// |    Cork is distributed in the hope that it will be useful,
// |    but WITHOUT ANY WARRANTY; without even the implied warranty of
// |    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// |    GNU Lesser General Public License for more details.
// |
// |    You should have received a copy
// |    of the GNU Lesser General Public License
// |    along with Cork.  If not, see <http://www.gnu.org/licenses/>.
// +-------------------------------------------------------------------------
#include "cork.h"
#include "file_formats/files.h"
#include "mesh/mesh.h"

namespace
{

typedef Cork::Mesh<Cork::CorkVertex, Cork::CorkTriangle> CorkMesh;

template <typename TVert1, typename TTri1, typename TVert2, typename TTri2>
void convertRawMesh(
    const Cork::RawMesh<TVert1, TTri1>& from,
    Cork::RawMesh<TVert2, TTri2>& to)
{
    to.triangles =
        std::vector<TTri2>(from.triangles.begin(), from.triangles.end());
    to.vertices =
        std::vector<TVert2>(from.vertices.begin(), from.vertices.end());
}

} // namespace

Cork::CorkVertex::CorkVertex()
{}

Cork::CorkVertex::CorkVertex(const MinimalVertexData& from)
    : MinimalVertexData(from)
    , RemeshVertexData()
    , BoolVertexData()
{}

void Cork::CorkVertex::isctInterpolate(
    const Cork::CorkVertex& v0,
    const Cork::CorkVertex& v1)
{
    double a0 = len(v1.pos - pos);
    double a1 = len(v0.pos - pos);
    if(a0 + a1 == 0.0)
        a0 = a1 = 0.5; // safety
    double sum = a0 + a1;
    a0 /= sum;
    a1 /= sum;
}

Cork::CorkTriangle::CorkTriangle()
{}

Cork::CorkTriangle::CorkTriangle(const MinimalTriangleData& from)
    : MinimalTriangleData(from)
    , BoolTriangleData()
{}

void Cork::CorkTriangle::subdivide(
    SubdivideTriInput<Cork::CorkVertex, Cork::CorkTriangle> input)
{
    bool_alg_data = input.pt->bool_alg_data;
}

Cork::CorkRawMesh Cork::loadMesh(const std::string& filename)
{
    Files::FileMesh filemesh;
    if(Files::readTriMesh(filename, &filemesh) > 0)
        throw std::runtime_error("Unable to load in " + filename);
    CorkRawMesh rawMesh;
    convertRawMesh(filemesh, rawMesh);
    return rawMesh;
}

void Cork::saveMesh(const std::string& filename, const Cork::CorkRawMesh& mesh)
{
    Files::FileMesh fileMesh;
    convertRawMesh(mesh, fileMesh);
    if(Files::writeTriMesh(filename, &fileMesh) > 0)
        throw std::runtime_error("Unable to write to " + filename);
}

bool Cork::isSolid(const CorkRawMesh& rawMesh)
{
    CorkMesh mesh(rawMesh);
    bool solid = true;
    if(mesh.isSelfIntersecting())
    {
        CORK_ERROR("isSolid() was given a self-intersecting mesh");
        solid = false;
    }
    if(!mesh.isClosed())
    {
        CORK_ERROR("isSolid() was given a non-closed mesh");
        solid = false;
    }
    return solid;
}

Cork::CorkRawMesh Cork::computeUnion(
    const Cork::CorkRawMesh& in0,
    const Cork::CorkRawMesh& in1,
    const int nTries)
{
    CorkMesh m0(in0);
    CorkMesh m1(in1);
    m0.boolUnion(m1, nTries);
    return m0.raw();
}

Cork::CorkRawMesh Cork::computeDifference(
    const CorkRawMesh& in0,
    const CorkRawMesh& in1,
    const int nTries)
{
    CorkMesh m0(in0);
    CorkMesh m1(in1);
    m0.boolDiff(m1, nTries);
    return m0.raw();
}

Cork::CorkRawMesh Cork::computeIntersection(
    const CorkRawMesh& in0,
    const CorkRawMesh& in1,
    const int nTries)
{
    CorkMesh m0(in0);
    CorkMesh m1(in1);
    m0.boolIsct(m1, nTries);
    return m0.raw();
}

Cork::CorkRawMesh Cork::computeSymmetricDifference(
    const Cork::CorkRawMesh& in0,
    const Cork::CorkRawMesh& in1,
    const int nTries)
{
    CorkMesh m0(in0);
    CorkMesh m1(in1);
    m0.boolXor(m1, nTries);
    return m0.raw();
}

Cork::CorkRawMesh Cork::resolveIntersections(
    const Cork::CorkRawMesh& in0,
    const Cork::CorkRawMesh& in1,
    const int nTries)
{
    CorkMesh m0(in0);
    CorkMesh m1(in1);
    m0.disjointUnion(m1);
    m0.resolveIntersections(nTries);
    return m0.raw();
}
